<?php
/**
 * Team Corrections Admin Interface
 * Manages the admin interface for team name and logo corrections
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class ApiFootballTeamCorrectionsAdmin {
    
    private $team_corrections;
    private $api_client;
    
    public function __construct() {
        $this->team_corrections = new ApiFootballTeamCorrections();
        $this->api_client = new ApiFootballClient();
    }
    
    /**
     * Display the team corrections admin page
     */
    public function displayPage() {
        // Handle form submissions
        if (isset($_POST['action'])) {
            $this->handleFormSubmissions();
        }
        
        $current_tab = isset($_GET['tab']) ? $_GET['tab'] : 'manage';
        
        ?>
        <div class="wrap api-football-team-corrections">
            <h1><?php _e('Team Corrections', 'api-football'); ?></h1>
            
            <nav class="nav-tab-wrapper">
                <a href="?page=api-football-team-corrections&tab=manage" class="nav-tab <?php echo $current_tab == 'manage' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Manage Corrections', 'api-football'); ?>
                </a>
                <a href="?page=api-football-team-corrections&tab=discover" class="nav-tab <?php echo $current_tab == 'discover' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Discover Teams', 'api-football'); ?>
                </a>
                <a href="?page=api-football-team-corrections&tab=import" class="nav-tab <?php echo $current_tab == 'import' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Import/Export', 'api-football'); ?>
                </a>
            </nav>
            
            <?php
            switch ($current_tab) {
                case 'discover':
                    $this->displayDiscoverTab();
                    break;
                case 'import':
                    $this->displayImportTab();
                    break;
                default:
                    $this->displayManageTab();
                    break;
            }
            ?>
        </div>
        <?php
    }
    
    /**
     * Display manage corrections tab
     */
    private function displayManageTab() {
        $corrections = $this->team_corrections->getAllCorrections();
        ?>
        <div class="tab-content">
            <h2><?php _e('Team Corrections Management', 'api-football'); ?></h2>
            <p><?php _e('Here you can manually correct team names and logos that are wrong or missing from the API.', 'api-football'); ?></p>
            
            <!-- Add New Correction -->
            <div class="postbox">
                <h3><?php _e('Add New Correction', 'api-football'); ?></h3>
                <div class="inside">
                    <form method="post" action="">
                        <?php wp_nonce_field('api_football_team_correction', 'team_correction_nonce'); ?>
                        <input type="hidden" name="action" value="add_correction">
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php _e('Team ID', 'api-football'); ?></th>
                                <td>
                                    <input type="number" name="team_id" required class="regular-text">
                                    <p class="description"><?php _e('The team ID from the API (find this in the Discover Teams tab)', 'api-football'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Original Name', 'api-football'); ?></th>
                                <td>
                                    <input type="text" name="original_name" required class="regular-text">
                                    <p class="description"><?php _e('The original name from the API', 'api-football'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Corrected Name', 'api-football'); ?></th>
                                <td>
                                    <input type="text" name="corrected_name" class="regular-text">
                                    <p class="description"><?php _e('The correct name to display (leave empty to use original)', 'api-football'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('Custom Logo URL', 'api-football'); ?></th>
                                <td>
                                    <input type="url" name="custom_logo_url" class="regular-text">
                                    <p class="description"><?php _e('URL of the correct logo (leave empty to use API logo)', 'api-football'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php _e('League ID', 'api-football'); ?></th>
                                <td>
                                    <input type="number" name="league_id" class="regular-text">
                                    <p class="description"><?php _e('Specific league ID (leave empty for global correction)', 'api-football'); ?></p>
                                </td>
                            </tr>
                        </table>
                        
                        <?php submit_button(__('Add Correction', 'api-football')); ?>
                    </form>
                </div>
            </div>
            
            <!-- Existing Corrections -->
            <div class="postbox">
                <h3><?php _e('Existing Corrections', 'api-football'); ?></h3>
                <div class="inside">
                    <?php if (empty($corrections)): ?>
                        <p><?php _e('No team corrections found. Add corrections above or use the Discover Teams tab to find teams that need corrections.', 'api-football'); ?></p>
                    <?php else: ?>
                        <table class="wp-list-table widefat fixed striped">
                            <thead>
                                <tr>
                                    <th><?php _e('Team ID', 'api-football'); ?></th>
                                    <th><?php _e('Original Name', 'api-football'); ?></th>
                                    <th><?php _e('Corrected Name', 'api-football'); ?></th>
                                    <th><?php _e('Custom Logo', 'api-football'); ?></th>
                                    <th><?php _e('League ID', 'api-football'); ?></th>
                                    <th><?php _e('Actions', 'api-football'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($corrections as $correction): ?>
                                    <tr>
                                        <td><?php echo esc_html($correction['team_id']); ?></td>
                                        <td><?php echo esc_html($correction['original_name']); ?></td>
                                        <td><?php echo esc_html($correction['corrected_name'] ?: __('(same as original)', 'api-football')); ?></td>
                                        <td>
                                            <?php if ($correction['custom_logo_url']): ?>
                                                <img src="<?php echo esc_url($correction['custom_logo_url']); ?>" alt="Logo" style="width: 30px; height: 30px;">
                                            <?php else: ?>
                                                <?php _e('(use API logo)', 'api-football'); ?>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo esc_html($correction['league_id'] ?: __('Global', 'api-football')); ?></td>
                                        <td>
                                            <form method="post" style="display: inline;">
                                                <?php wp_nonce_field('api_football_team_correction', 'team_correction_nonce'); ?>
                                                <input type="hidden" name="action" value="delete_correction">
                                                <input type="hidden" name="correction_id" value="<?php echo esc_attr($correction['id']); ?>">
                                                <button type="submit" class="button button-small" onclick="return confirm('<?php _e('Are you sure you want to delete this correction?', 'api-football'); ?>')">
                                                    <?php _e('Delete', 'api-football'); ?>
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Display discover teams tab
     */
    private function displayDiscoverTab() {
        $teams = array();
        $options = get_option('api_football_options', array());
        $league_id = $options['league_id'] ?? '286';
        
        // Get teams from current standings
        $standings = $this->api_client->getStandings($league_id);
        if ($standings) {
            $teams = $this->team_corrections->getTeamsFromApiData($standings);
        }
        
        ?>
        <div class="tab-content">
            <h2><?php _e('Discover Teams', 'api-football'); ?></h2>
            <p><?php _e('This shows all teams from your current league. You can quickly add corrections for teams with missing or incorrect information.', 'api-football'); ?></p>
            
            <div class="postbox">
                <h3><?php _e('Teams in Current League', 'api-football'); ?> (<?php echo esc_html($league_id); ?>)</h3>
                <div class="inside">
                    <?php if (empty($teams)): ?>
                        <p><?php _e('No teams found. Make sure your API key is configured and test the connection in the main settings.', 'api-football'); ?></p>
                    <?php else: ?>
                        <table class="wp-list-table widefat fixed striped">
                            <thead>
                                <tr>
                                    <th><?php _e('Team ID', 'api-football'); ?></th>
                                    <th><?php _e('Logo', 'api-football'); ?></th>
                                    <th><?php _e('Name', 'api-football'); ?></th>
                                    <th><?php _e('Status', 'api-football'); ?></th>
                                    <th><?php _e('Actions', 'api-football'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($teams as $team): ?>
                                    <?php
                                    $existing_correction = $this->team_corrections->getTeamCorrection($team['id'], $league_id);
                                    $has_logo = !empty($team['logo']) && filter_var($team['logo'], FILTER_VALIDATE_URL);
                                    ?>
                                    <tr>
                                        <td><?php echo esc_html($team['id']); ?></td>
                                        <td>
                                            <?php if ($has_logo): ?>
                                                <img src="<?php echo esc_url($team['logo']); ?>" alt="Logo" style="width: 30px; height: 30px;" onerror="this.style.display='none'; this.nextElementSibling.style.display='inline';">
                                                <span style="display: none; color: red;">❌</span>
                                            <?php else: ?>
                                                <span style="color: red;">❌ <?php _e('Missing', 'api-football'); ?></span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo esc_html($team['name']); ?></td>
                                        <td>
                                            <?php if ($existing_correction): ?>
                                                <span style="color: green;">✅ <?php _e('Has correction', 'api-football'); ?></span>
                                            <?php elseif (!$has_logo): ?>
                                                <span style="color: orange;">⚠️ <?php _e('Needs logo', 'api-football'); ?></span>
                                            <?php else: ?>
                                                <span style="color: gray;">➖ <?php _e('OK', 'api-football'); ?></span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <button type="button" class="button button-small" onclick="fillCorrectionForm(<?php echo esc_attr($team['id']); ?>, '<?php echo esc_js($team['name']); ?>', '<?php echo esc_js($team['logo']); ?>', <?php echo esc_attr($league_id); ?>)">
                                                <?php _e('Add Correction', 'api-football'); ?>
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <script>
        function fillCorrectionForm(teamId, teamName, teamLogo, leagueId) {
            // Switch to manage tab and fill form
            const manageTab = document.querySelector('a[href*="tab=manage"]');
            if (manageTab) {
                window.location.href = manageTab.href + '&fill_team_id=' + teamId + '&fill_team_name=' + encodeURIComponent(teamName) + '&fill_league_id=' + leagueId;
            }
        }
        </script>
        <?php
        
        // Auto-fill form if parameters are present
        if (isset($_GET['fill_team_id'])) {
            ?>
            <script>
            document.addEventListener('DOMContentLoaded', function() {
                const teamIdField = document.querySelector('input[name="team_id"]');
                const originalNameField = document.querySelector('input[name="original_name"]');
                const leagueIdField = document.querySelector('input[name="league_id"]');
                
                if (teamIdField) teamIdField.value = '<?php echo esc_js($_GET['fill_team_id']); ?>';
                if (originalNameField) originalNameField.value = '<?php echo esc_js($_GET['fill_team_name'] ?? ''); ?>';
                if (leagueIdField) leagueIdField.value = '<?php echo esc_js($_GET['fill_league_id'] ?? ''); ?>';
                
                // Scroll to form
                const form = document.querySelector('form input[name="action"][value="add_correction"]').closest('form');
                if (form) {
                    form.scrollIntoView({ behavior: 'smooth' });
                }
            });
            </script>
            <?php
        }
    }
    
    /**
     * Display import/export tab
     */
    private function displayImportTab() {
        ?>
        <div class="tab-content">
            <h2><?php _e('Import/Export Team Corrections', 'api-football'); ?></h2>
            
            <!-- Export Section -->
            <div class="postbox">
                <h3><?php _e('Export Corrections', 'api-football'); ?></h3>
                <div class="inside">
                    <p><?php _e('Export your current team corrections as a JSON file for backup or sharing.', 'api-football'); ?></p>
                    <form method="post">
                        <?php wp_nonce_field('api_football_team_correction', 'team_correction_nonce'); ?>
                        <input type="hidden" name="action" value="export_corrections">
                        <?php submit_button(__('Export Corrections', 'api-football'), 'secondary'); ?>
                    </form>
                </div>
            </div>
            
            <!-- Import Section -->
            <div class="postbox">
                <h3><?php _e('Import Corrections', 'api-football'); ?></h3>
                <div class="inside">
                    <p><?php _e('Import team corrections from a JSON file. This will add to existing corrections.', 'api-football'); ?></p>
                    <form method="post" enctype="multipart/form-data">
                        <?php wp_nonce_field('api_football_team_correction', 'team_correction_nonce'); ?>
                        <input type="hidden" name="action" value="import_corrections">
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php _e('JSON File', 'api-football'); ?></th>
                                <td>
                                    <input type="file" name="import_file" accept=".json" required>
                                    <p class="description"><?php _e('Select a JSON file containing team corrections.', 'api-football'); ?></p>
                                </td>
                            </tr>
                        </table>
                        
                        <?php submit_button(__('Import Corrections', 'api-football')); ?>
                    </form>
                    
                    <h4><?php _e('JSON Format Example:', 'api-football'); ?></h4>
                    <pre style="background: #f1f1f1; padding: 15px; overflow-x: auto;"><code>[
  {
    "team_id": 12360,
    "original_name": "Brodarac",
    "corrected_name": "FK Brodarac",
    "custom_logo_url": "https://example.com/logo.png",
    "league_id": 286
  }
]</code></pre>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Handle form submissions
     */
    private function handleFormSubmissions() {
        if (!isset($_POST['team_correction_nonce']) || !wp_verify_nonce($_POST['team_correction_nonce'], 'api_football_team_correction')) {
            wp_die(__('Security check failed', 'api-football'));
        }
        
        switch ($_POST['action']) {
            case 'add_correction':
                $this->handleAddCorrection();
                break;
            case 'delete_correction':
                $this->handleDeleteCorrection();
                break;
            case 'export_corrections':
                $this->handleExportCorrections();
                break;
            case 'import_corrections':
                $this->handleImportCorrections();
                break;
        }
    }
    
    /**
     * Handle adding a correction
     */
    private function handleAddCorrection() {
        $data = array(
            'team_id' => intval($_POST['team_id']),
            'original_name' => sanitize_text_field($_POST['original_name']),
            'corrected_name' => sanitize_text_field($_POST['corrected_name']),
            'custom_logo_url' => esc_url_raw($_POST['custom_logo_url']),
            'league_id' => !empty($_POST['league_id']) ? intval($_POST['league_id']) : null
        );
        
        if ($this->team_corrections->saveCorrection($data)) {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-success is-dismissible"><p>' . __('Team correction saved successfully!', 'api-football') . '</p></div>';
            });
        } else {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error is-dismissible"><p>' . __('Error saving team correction.', 'api-football') . '</p></div>';
            });
        }
    }
    
    /**
     * Handle deleting a correction
     */
    private function handleDeleteCorrection() {
        $id = intval($_POST['correction_id']);
        
        if ($this->team_corrections->deleteCorrection($id)) {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-success is-dismissible"><p>' . __('Team correction deleted successfully!', 'api-football') . '</p></div>';
            });
        } else {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error is-dismissible"><p>' . __('Error deleting team correction.', 'api-football') . '</p></div>';
            });
        }
    }
    
    /**
     * Handle exporting corrections
     */
    private function handleExportCorrections() {
        $json = $this->team_corrections->exportCorrections();
        $filename = 'api-football-team-corrections-' . date('Y-m-d') . '.json';
        
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . strlen($json));
        
        echo $json;
        exit;
    }
    
    /**
     * Handle importing corrections
     */
    private function handleImportCorrections() {
        if (!isset($_FILES['import_file']) || $_FILES['import_file']['error'] !== UPLOAD_ERR_OK) {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error is-dismissible"><p>' . __('Error uploading file.', 'api-football') . '</p></div>';
            });
            return;
        }
        
        $file_content = file_get_contents($_FILES['import_file']['tmp_name']);
        $result = $this->team_corrections->importCorrections($file_content);
        
        if ($result['success']) {
            add_action('admin_notices', function() use ($result) {
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html($result['message']) . '</p></div>';
            });
        } else {
            add_action('admin_notices', function() use ($result) {
                echo '<div class="notice notice-error is-dismissible"><p>' . esc_html($result['message']) . '</p></div>';
            });
        }
    }
}